;=========================================================================
; GB222.ASM - Demonstrates several programming techniques that are
;	      appropriate for controlling the Hercules InColor Card.
;
; Highlights include:
;	      - A complete equate block for the GB222.
;	      -	Testing for the presence of the InColor Card.
;	      -	Changing to graphics and text modes.
;	      -	Initializing the color palette.
;	      - Setting up write modes and color registers.
;	      - Drawing colored lines across a random background.
;	      - Masking planes to write to and display specific planes.
;	      - Using latch protect to copy an object onto an unknown
;	        background.
;=========================================================================
; GB222 equates

dmc_port	equ	3B8h		; Display Mode Control Port
status_port	equ	3BAh		; Display Status Port
config_port	equ	3BFh		; Hercules Configuration Port

index_reg	equ	3B4h		; 6845 index register
data_reg	equ	3B5h		; 6845 data register

xmode_reg		equ	14h	; bit 0    char generator (ROM,RAM)
					; bit 1    Matrix width (9 or 8)
					; bit 2    RamFont mode (4k or 48k)
					; bit 3-7  unused

underscore_reg		equ	15h	; bit 0-3  underscore position
					; bit 4-7  underscore color

overstrike_reg		equ	16h	; bit 0-3  overstrike position
					; bit 4-7  overstrike color

exception_reg		equ	17h	; bit 0-3  cursor color
					; bit 4    palette enable/disable
					; bit 5    normal/alternate attributes
					; bit 6-7  unused

plane_mask_reg		equ	18h	; bit 0-3  display planes (visual
					;          mask)
					; bit 4-7  freeze planes (operation
					;          mask)

r_w_control_reg		equ	19h	; bit 0-3  don't care planes
					; bit 4-5  write mode
					; bit 6    mask polarity
					; bit 7    unused

r_w_color_reg		equ	1Ah	; bit 0-3  foreground color value
					; bit 4-7  background color value

latch_protect_reg	equ	1Bh	; bit 0 -  protect pixel 0 (rightmost)
					; bit 1 -  protect pixel 1
					; bit 2 -  protect pixel 2
					; bit 3 -  protect pixel 3
					; bit 4 -  protect pixel 4
					; bit 5 -  protect pixel 5
					; bit 6 -  protect pixel 6
					; bit 7 -  protect pixel 7 (leftmost)

palette_reg		equ	1Ch	; bits 0-7, read resets palette index
					; write, auto increments to next index
					; bit 0    blue
					; bit 1    green
					; bit 2    red
					; bit 3    secondary blue
					; bit 4    secondary green
					; bit 5    secondary red


id_mask		equ	01110000b	; to detect the presence of
id_code		equ	01010000b	; a GB222

diag		equ	0		; configuration port options
half		equ	1
full		equ	3
			   	
grph		equ	00000010b	; equates for dmc_port
text		equ	00000000b
screen_on	equ	00001000b
screen_off	equ	00000000b
blinker_on	equ	00100000b
blinker_off	equ	00000000b
graph_page_1	equ	10000000b
graph_page_0	equ	00000000b

ROM_text	equ	00000000b	; equates for xMode Register
RAM_text	equ	00000001b
nine_wide	equ	00000000b
eight_wide	equ	00000010b
font_4k		equ	00000000b
font_48k	equ	00000100b

enable_palette	equ	00010000b	; equates for Exception Register
disable_palette equ	00000000b
alternate_att	equ	00000000b
normal_att	equ	00100000b

display_all	equ	00001111b	; equates for Plane Mask Register
display_none	equ	00000000b
display_0   	equ	00000001b
display_1   	equ	00000010b
display_2   	equ	00000100b
display_3   	equ	00001000b
freeze_none	equ	00000000b
freeze_all	equ	11110000b
freeze_0  	equ	00010000b
freeze_1  	equ	00100000b
freeze_2  	equ	01000000b
freeze_3  	equ	10000000b

care_all	equ	00000000b	; equates for r_w_control register
no_care_all	equ	00001111b
write_mode_0	equ	00000000b
write_mode_1	equ	00010000b
write_mode_2	equ	00100000b
write_mode_3	equ	00110000b
set_if_equal	equ	00000000b
clear_if_equal	equ	01000000b

;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
; SEND - Macro to initialize a single 6845 register
; ON ENTRY - two arguments, the register to initialize and the value to use.
;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
Send	macro	reg, value
	mov	dx,index_reg
	mov	al,reg
	out	dx,al
	inc	dx
	mov	al,value
	out	dx,al
	endm

Wait_for_key	macro			; pauses execution until keypress
	mov	ah,0
	int	16h
	endm

;======================================================================

cseg	segment	public 'CODE'
	assume	cs:cseg,ds:dseg,ss:stack,es:nothing
start	proc	far

	mov	ax,dseg
	mov	ds,ax

	call	test_board		; verify that an InColor Card
					; is installed
	jae	board_ok

	lea	dx,board_err_msg	; if not, display error message
	mov	ah,9
	int	21h

	mov	ax,4C01h		; and quit
	int	21h

board_ok:				; if so ...

	call	construct_y_table	; set up shop

	mov	dx,config_port		; Set the InColor Card to the
	mov	al,half			; HALF configuration
	out	dx,al

	call	initialize_palette	; program, and enable palette

	call	gmode			; set graphics mode

; Set up GB222

	Send	plane_mask_reg,freeze_none+display_all
	
	Send	r_w_control_reg,clear_if_equal+write_mode_0+care_all

	Send	r_w_color_reg,00000100b	  ; black back, red fore

; Write a color to each of the four quadrants, starting at the upper left

	mov	ax,0b000h			; point ES at screen buffer
	mov	es,ax

	xor	bx,bx				; start from scan line 0
	mov	al,0ffh				; write all high bits

	mov	cx,174				; 174 lines, 1/2 the height 
fill_quad:					; of the screen
	push	cx
	push	bx
	shl	bx,1				; table contains word values
	mov	di,cs:ytable[bx]

	mov	cx,45				; write 45 bytes, 1/2 the
rep	stosb					; width of the screen
	pop	bx
	inc	bx
	pop	cx
	loop	fill_quad

	wait_for_key
	
; fill upper right quadrant

	Send	r_w_color_reg,00000010b		; black back green fore

	xor	bx,bx
	mov	al,0ffh

	mov	cx,174
fill_quad_2:
	push	cx
	push	bx
	shl	bx,1
	mov	di,cs:ytable[bx]
	add	di,45				; paint the right half

	mov	cx,45
rep	stosb
	pop	bx
	inc	bx
	pop	cx
	loop	fill_quad_2

	wait_for_key

; fill lower left quadrant

	Send	r_w_color_reg,00000001b		; black back, blue fore

	mov	bx,174				; start at the 174th line
	mov	al,0ffh

	mov	cx,174
fill_quad_3:
	push	cx
	push	bx
	shl	bx,1
	mov	di,cs:ytable[bx]

	mov	cx,45
rep	stosb
	pop	bx
	inc	bx
	pop	cx
	loop	fill_quad_3

	wait_for_key

;fill lower right quadrant

	Send	r_w_color_reg,00000110b		; black back, brown fore

	mov	bx,174
	mov	al,0ffh

	mov	cx,174
fill_quad_4:
	push	cx
	push	bx
	shl	bx,1
	mov	di,cs:ytable[bx]
	add	di,45

	mov	cx,45
rep	stosb
	pop	bx
	inc	bx
	pop	cx
	loop	fill_quad_4

	wait_for_key

; Write a green line across two different color fields 

	Send	r_w_color_reg,00100010b ; green back, green fore
	Send	r_w_control_reg,set_if_equal+write_mode_1

	mov	bx,185			; write to scan line 185
	shl	bx,1
	mov	di,ytable[bx]
	mov	cx,90			; write the full width of screen
	mov	al,-1		        ; write all high bits
write_line:
	or	es:[di],al		; using the OR instruction
	inc	di
	loop	write_line

	wait_for_key

; use same logical settings to write a broken green line

	mov	bx,190			; write this line at scan line 190
	shl	bx,1
	mov	di,ytable[bx]
	mov	cx,90
	mov	al,0fh			; use mask for broken line (00001111b)
write_line_2:
	or	es:[di],al
	inc	di
	loop	write_line_2

	wait_for_key

; write a broken red line across two color fields

	Send	r_w_color_reg,01000100b ; red back, red fore

	mov	bx,200
	shl	bx,1
	mov	di,ytable[bx]
	mov	cx,90
	mov	al,0f0h
write_line_3:
	or	es:[di],al
	inc	di
	loop	write_line_3

	wait_for_key

; write a red line across two color fields using planes mask method

; we want to be writing all high bits
	Send	r_w_color_reg,11111111b	; white back, white fore

; freeze all planes except red
	Send	plane_mask_reg,freeze_0+freeze_1+freeze_3+display_all

	Send	r_w_control_reg,set_if_equal+write_mode_0+care_all

	mov	bx,210
	shl	bx,1
	mov	di,ytable[bx]
	mov	cx,90
	mov	al,-1
write_line_4:
	mov	es:[di],al
	inc	di
	loop	write_line_4

	Send	plane_mask_reg,freeze_none+display_all

	wait_for_key

; demonstrates masking off visual planes

	Send	plane_mask_reg,freeze_none+display_0

	wait_for_key

	Send	plane_mask_reg,freeze_none+display_1

	wait_for_key

	Send	plane_mask_reg,freeze_none+display_2

	wait_for_key

	Send	plane_mask_reg,freeze_none+display_3

	wait_for_key

	Send	plane_mask_reg,freeze_none+display_all

	wait_for_key

; the following code demonstrates using latch protect to write onto
; an unknown background

; write a pattern of colored pixels on a color field

	Send	r_w_control_reg,set_if_equal+write_mode_0

	Send	r_w_color_reg,01000010b	; red back, green fore

	mov	al,42h			; the mask to write (01000010)

	mov	bx,(174/2)-10
	shl	bx,1
	mov	cx,20
write_block:
	push	cx
	mov	di,ytable[bx]
	add	di,(45/2)-2
	mov	cx,5
rep	stosb
	add	bx,2
	pop	cx
	loop	write_block

	wait_for_key

	mov	bx,(174/2)-10
	shl	bx,1
	mov	di,ytable[bx]
	add	di,(45/2)-2		; target byte in DI

	mov	object_position,di

	Send	r_w_control_reg,set_if_equal+write_mode_3+care_all

	Send	r_w_color_reg,00100010b	; green back, green fore

	mov	ah,es:[di]		; scan the source with backg set

	mov	al,latch_protect_reg
	mov	dx,index_reg			; protect the object outline
	out	dx,ax				; (not background)

; get dest into DI
	mov	bx,(174/2)-10
	add	bx,174
	shl	bx,1
	mov	di,ytable[bx]
	add	di,(45/2)-2

	mov	al,es:[di]		; compose object by mixing destination
					; pattern with object already
					; in the latch 

	mov	al,-1			; write out whole latch
					; don't complement any pixel
	mov	cx,20
write_block_1:
	push	cx

	mov	di,ytable[bx]
	add	di,(45/2)-2
	mov	cx,5
rep	stosb				; write object in new destination
	add	bx,2
	pop	cx
	loop	write_block_1

	wait_for_key

; get another destination

	mov	bx,(174/2)-10
	add	bx,174
	shl	bx,1
	mov	di,ytable[bx]
	add	di,(45/2)-2
	add	di,45

	mov	al,es:[di]		; compose object by mixing destination
					; pattern with object already
					; in the latch 

	mov	al,-1			; to write all high bits

	mov	cx,20
write_block_2:
	push	cx
	mov	di,ytable[bx]
	add	di,(45/2)-2
	add	di,45
	mov	cx,5
rep	stosb				; write object in new destination
	add	bx,2
	pop	cx
	loop	write_block_2

; unprotect the latch

	SEND	latch_protect_reg,0	

	wait_for_key

	call	reset_222

	call	tmode

	mov	ax,4C00h
	int	21h			; return to DOS

start	endp

;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
; INITIALIZE_PALETTE - loads the palette_table data into the GB222 palette
;		       registers and enables the palette.
;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
initialize_palette	proc	near

	mov	dx,index_reg
	mov	al,palette_reg		; point at GB222 palette registers
	out	dx,al
	inc	dx			; increment to 6845 data register
	in	al,dx			; reset the palette pointer

	cld
	mov	cx,16			; 16 registers in the palette
	lea	si,palette_table	; our table of colors
load_palette:
	lodsb				; get value into AL
	out	dx,al			; put it into register, palette
					; auto increments to next index
	loop	load_palette

	mov	dx,index_reg
	mov	al,exception_reg
	out	dx,al
	inc	dx
	mov	al,enable_palette + normal_att
	out	dx,al

	ret

initialize_palette	endp

;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
; GMODE - Programs the 6845 CRT controller for graphics mode.
;         The active page for both writing and display is set to the
;         default value of page 0.
;
; ON ENTRY - no parameters.
;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
gmode	proc	near

	mov	al,grph		; value for the dmc_port	
	lea	si,gtable	; table of values for the 6845
	mov	bx,0000		; word value used to clear graphics buffer		
	mov	cx,4000h	; which is 4000h words long

	call	setmd

	ret

gmode	endp

;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
; TMODE - Programs the 6845 and control register to produce text mode.
;
; ON ENTRY - no parameters.
;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
tmode	proc	near

	mov	al,text		; value for dmc_port
	lea	si,ttable	; table of values for 6845
	mov	bx,720h		; word value used to clear text buffer
	mov	cx,2000		; which is 2000 words long

	call	setmd

	ret

tmode	endp

;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
; SETMD - Sets the mode to either graphics or text depending on the
;         contents of AL.
; ON ENTRY - AL contains the value for the DMC_Port (text or graphics).
;            SI points to a table of values to program into the 6845.
;	     CX contains the length of the screen buffer in words.
;	     BX contains the word value to use to blank the screen.
;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
setmd	proc	near

	push	ds
	push	es
	push	ax
	push	bx
	push	cx

; change mode but without scrn_on
	mov	dx,dmc_port
	out	dx,al

; intialize the 6845

	mov	dx,index_reg
	mov	cx,12		;12 parameters to
				;be output
	xor	ah,ah		;starting from
				;reg. 0
parms:	mov	al,ah
	out	dx,al		;output register
				;number
	cld

	inc	dx
	lodsb
	out	dx,al		;output data

	inc	ah		;next value
	dec	dx
	loop	parms

	pop	cx		;clear the buffer
	mov	ax,0b000h

	mov	es,ax
	xor	di,di
	pop	ax
	rep	stosw

; scrn_on, page 0

	mov	dx,dmc_port
	pop	ax
	add	al,screen_on
	out	dx,al

	pop	es
	pop	ds

	ret

setmd	endp

;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
; TEST_BOARD - Tests for the presence of a Hercules InColor Card
;
; RETURNS -  carry flag set if a GB222 is not detected
;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * 
test_board	proc	near
	mov	ah,15			; check current video state
	int	10h

	cmp	al,7			; is it 80 x 25 "B&W" Card
	je	BIOS_mode_ok		; yes - continue

	jmp	test_failed		; no - quit

BIOS_mode_ok:		       	
	mov	dx,status_port		; record state 
	in	al,dx			
	and	al,80h			; save bit 7 for test
	mov	ah,al

	mov	cx,4000h
examine:
	in	al,dx			; take another reading
	and	al,80h			; again, save bit seven
	cmp	al,ah
	jne	hgc_ok			; if bit 7 changes, then it
	loop	examine			; is a Hercules Card
    
	jmp	test_failed		; after this long it must be
					; something else - quit.
hgc_ok:
	in	al,dx			; test for GB222
	and	al,id_mask		; clear all but bits 4, 5, and 6

	cmp	al,id_code		; test ID bits
	jne	test_failed		; exit if failed - not a GB222

	clc				; clear carry flag
	ret				; normal return

test_failed:
	stc				; set the carry flag
	ret				; error return

test_board	endp

;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
; RESET_222 - initializes all propriatary registers to their power-on
;	      states.
;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
reset_222	proc	near

	Send	xMode_reg,0
	Send	underscore_reg,0
	Send	overstrike_reg,0
	Send	exception_reg,normal_att
	Send	plane_mask_reg,freeze_none+display_all
	Send	r_w_control_reg,clear_if_equal+write_mode_0+care_all
	Send	r_w_color_reg,00001111b
	Send	latch_protect_reg,0

	ret
reset_222	endp

;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
; CONSTRUCT_Y_TABLE - Sets up a table of Y coordinate offsets
;* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
construct_y_table	proc	near
	push	ds
	pop	es

	lea	di,ytable
	cld

	xor	ax,ax
	mov	cx,87			; there are 4 frames of 87 lines each
init_ytable:				; in the Hercules graphics mode
	stosw
	add	ax,2000h		; each frame contains 2000h bytes
	stosw
	add	ax,2000h
	stosw
	add	ax,2000h
	stosw
	sub	ax,6000h-90

	loop	init_ytable

	ret

construct_y_table	endp 

cseg	ends

dseg	segment public 'DATA'
	
gtable	   	db	35h,2dh,2eh,07h		; Recommended graphics
	   	db	5bh,02h,57h,57h		; mode values for the 
	   	db	02h,03h,00h,00h		; 6845
	   
ttable	 	db	61h,50h,52h,0fh		; Recommended text mode
	 	db	19h,06h,19h,19h		; values for the 6845
	 	db	02h,0dh,0bh,0ch

palette_table	db	0
		db	1
		db	2
		db	3
		db	4
		db	5
		db     20			; a better brown
		db	7
		db	8
		db	9
		db     10
		db     11
		db     12
		db     13
		db     14
		db     15

board_err_msg	db	0dh,0ah,' * ERROR * - Hercules InColor Card not'
		db	' present, or not presently activated.',0dh,0ah,'$'

object_position dw	?

ytable		dw	348 dup (?)

dseg 	ends

stack	segment	stack 'STACK'

	dw	64 dup (?)

stack	ends

	end	start


